(*******************************************************************************
**
**
**  Component in this unit originally written by and copyright
**                     by Andy Strong - Compuserve ID   : 100716,3015
**                                      Internet Address: andrews@nbaqsl.co.uk
**
**  Released into the public domain 09/05/96
**  Updated: vs 1.1 -  20/12/96 - Changed name, fixed minor LOAD bug.
**
**  Controls:
**  This code may be used, modified, included in applications without any
**  license agreements as long as the disclaimers are accepted, and the
**  comments are left intact ( Some Hope! <g> ),
**
**  Disclaimer:
**  This software is released into the public domain on the strict understanding
**  that neither myself nor any associates or companies I work for have any
**  liability explictly or implied.
**
**
*******************************************************************************)
unit Resizwnd;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, ExtCtrls, StdCtrls, Gauges, Buttons, Menus;

type

(*******************************************************************************
**
**  FORMSIZELIMIT
**
**  Component designed to be dropped onto any form of an Application which
**  needs to have its resizeable border controlled as to its Maximum & Minimum
**  limits.
**  Note:  Certain property changes force a call to TWINCONTROLs RECREATEWND
**    method.  This will de-link the windows subclassed message handler and
**    render this component inoperable.  To circumvent this, ensure this is the
**    last component added to a sizeable form, or that this the last component
**    enabled on the form in question.
**
*******************************************************************************)
  { Define Exception for Min Height/Width below minimum }
  TMinSizeError = class(Exception);
  { Define Exception for Form BorderStyle not set to sizeable }
  TNotSizeableError = class(Exception);
  { Define Exception for Form not being owner of component }
  TNotOwnerError = class(Exception);
  { Define Exception for Already being one TFormSizeLimit component on Form }
  TDuplicationError = class(Exception);

  { Actual Class Definition }
  TFormSizeLimit = class(TComponent)
  private
    { Private declarations }
    FActive: boolean;                       { Whether Resize checks are wanted }
    FAutoTrack: boolean;                    { Auto track sizes during design time }
    FAbsMinHeight: word;                    { Absolute Minimum Height allowed }
    FAbsMinWidth: word;                     { Absolute Minimum Width allowed }
    FMaxWidth: word;                        { Max Allowed Width }
    FMinWidth: word;                        { Min Allowed Width }
    FMaxHeight: word;                       { Max Allowed Height }
    FMinHeight: word;                       { Min Allowed Height }
    FFormHandle: Hwnd;                      { Windows Handle of Form Component dropped on }
    FWndHandlerPtr: Pointer;                { Pointer to our Window Message Handler }
    FOldWndHandler: Pointer;                { Pointer to Old Window Message Handler }
    FMaxRange: TPoint;                      { X = Width, Y = Height }
    FMinRange: TPoint;                      { X = Width, Y = Height }

    procedure AdjustMinMaxMessage(MMPointer: Pointer); { Interacts with GETMINMAXINFO }
    procedure ChkMaxHeight(Setting: word);  { Check Range of Max Height }
    procedure ChkMaxWidth(Setting: word);   { Check Range of Max Width }
    procedure ChkMinHeight(Setting: word);  { Check Range of Min Height }
    procedure ChkMinWidth(Setting: word);   { Check Range of Min Width }
    procedure MakeNewWndProc(FormHandle: hWnd);{ Add new message handler }
    procedure NWndProc(var Messg: TMessage);{ New Window Message Handler }
    procedure SetNewSize(NewClWidth, NewClHeight: word ); { Set new sizes in Object Inspector }
    procedure SetOldWndProc(FormHandle: hWnd);  { Restore original message handler }
    procedure SetMinPoints;                 { Set Min TPoint Structures }
    procedure SetMaxPoints;                 { Set Min TPoint Structures }

  protected
    { Protected declarations }

  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Loaded; override;
    procedure SetMaxSize(MaxBounds: TPoint);{ Procedure to set Max Width/Height }
    procedure SetMinSize(MinBounds: TPoint);{ Procedure to set Min Width/Height }

  published
    { Published declarations }
    property Active: boolean read FActive write FActive;
    Property AutoTrack: boolean read FAutoTrack write FAutoTrack;
    property MaxHeight: word read FMaxHeight write ChkMaxHeight;
    property MaxWidth: word read FMaxWidth write ChkMaxWidth;
    property MinHeight: word read FMinHeight write ChkMinHeight;
    property MinWidth: word read FMinWidth write ChkMinWidth;
  end;


  procedure Register;

implementation

(*******************************************************************************
**
**  FORMSIZELIMIT
**
*******************************************************************************)
(*
**  Constructor for TFormSizeLimit objects
*)
constructor TFormSizeLimit.Create(AOwner: TComponent);
var
  i: word;
  ResizeCount: byte;
begin
  ResizeCount := 0;
  inherited Create(AOwner);             { Setup object originally }
  if (AOwner is TForm) then             { Only if Owner is a TForm object }
  begin
    with (AOwner as TForm) do
    begin
      if csDesigning in ComponentState then
      begin                             { If we're in design mode }
        for i := 0 to ComponentCount - 1 do{ Check if there is already one of us! }
          if Components[i] is TFormSizeLimit then
            inc(ResizeCount);
        if ResizeCount > 1 then   { One of them is us! }
           raise TDuplicationError.Create('Already a ResizeForm component on selected Form');
        if (BorderStyle <> bsSizeable) and
          not (csLoading in ComponentState) then{ Wrong Form type! }
          raise TNotSizeableError.Create('Form does not have BorderStyle of bsSizeable');
        { Set the current Minumum Height allowed to the original }
        FMinHeight := GetSystemMetrics(SM_CYMIN);
        FAbsMinHeight := FMinHeight;    { Set the Absolute Minimum Height }
        { Set the current Minumum Width allowed to the original }
        FMinWidth := GetSystemMetrics(SM_CXMIN);
        FAbsMinWidth := FMinWidth    ;  { Set the Absolute Minimum Width }
        MaxHeight := Height;            { Set the Current Max Height to the Form size }
        MaxWidth := Width;              { Set the Current Max Width to the Form size }
      end;
      FFormHandle := Handle;            { Save the Form Handle for Windows sub-classing }
      MakeNewWndProc(FFormHandle);      { Generate New Window message Handler }
    end;
  end
  else
    raise TNotOwnerError.Create('A Form MUST be the owner');
end;

(*
**  Loaded method overridden to set the windows message handler ranges to
**  the designed ranges.
*)
procedure TFormSizeLimit.Loaded;
begin
  inherited Loaded;           { Actually load design settings }
  SetMaxPoints;               { Set Max windows message handler ranges }
  SetMinPoints;               { Set Min windows message handler ranges }
end;

(*
**  Destructor for component.  This must relink the original windows message
**  handler in case the component is deleted from the form.
*)
destructor TFormSizeLimit.Destroy;
begin
  FAutoTrack := False;            { Ensure no more callback messages }
  SetOldWndProc(FFormHandle);     { Restore original windows message handler }
  if FWndHandlerPtr <> nil then
    FreeObjectInstance(FWndHandlerPtr); { Free up space used in creating window handler}
  inherited Destroy;
end;

(*
**  Check the maximum sizeable height is not set below the normal minimum height
*)
procedure TFormSizeLimit.ChkMaxHeight(Setting: word);
begin
  if (csDesigning in ComponentState) and (not (csLoading in ComponentState)) then
    if (Setting <= MinHeight) then
      raise TMinSizeError.Create('The Maximum Height must not be'#13 +
                                 'less than the minimum height');
  FMaxHeight := Setting;             { If run-time do not check }
  FMaxRange.Y := Setting;
end;

(*
**  Check the maximum sizeable width is not set below the normal minimum width
*)
procedure TFormSizeLimit.ChkMaxWidth(Setting: word);
begin
  if (csDesigning in ComponentState) and (not (csLoading in ComponentState)) then
    if (Setting <= MinWidth) then
      raise TMinSizeError.Create('The Maximum Width must not be'#13 +
                                 'less than the minimum width');
  FMaxWidth := Setting;            { If run-time do not check }
  FMaxRange.X := Setting;
end;

(*
**  Check the minimum sizeable height is not set below the normal minimum height
*)
procedure TFormSizeLimit.ChkMinHeight(Setting: word);
begin
  if (csDesigning in ComponentState) and (not (csLoading in ComponentState)) then
  begin
    if (Setting < FAbsMinHeight) then
      raise TMinSizeError.Create('The Minimum Height must not'#13'be less than caption height');
    if (Setting > FMaxHeight) then
      raise TMinSizeError.Create('The Minimum Height must not be'#13'greater than the Maximum height');
  end;
  FMinHeight := Setting;            { If run-time do not check }
  FMinRange.Y := Setting;
end;

(*
**  Check the minimum sizeable width is not set below the normal minimum width
*)
procedure TFormSizeLimit.ChkMinWidth(Setting: word);
begin
  if (csDesigning in ComponentState) and (not (csLoading in ComponentState)) then
  begin
    if (Setting < FAbsMinWidth) then
      raise TMinSizeError.Create('The Minimum Width must not be less'#13 +
                                 'than caption controls width');
    if (Setting > FMaxWidth) then
      raise TMinSizeError.Create('The Minimum width must not be'#13 +
                                 'greater than the Maximum Width');
  end;
  FMinWidth := Setting;            { If run-time do not check }
  FMinRange.X := Setting;
end;

(*
**  This procedure is a 'CallBack' procedure.  That is the window message
**  handling system Calls Back the new size of the form during designing.
**  This is needed to keep the Object Inspectors properties up to date with
**  the current Form size.
*)
procedure TFormSizeLimit.SetNewSize(NewClWidth, NewClHeight: word);
var
  NewWidth: word;
  NewHeight: word;
  SavActive: boolean;
begin
  { The CallBack routine from the windows resize message contains the Client
    Width & Height sizes.  Therefore we need to calculate the actual form
    dimensions. }
  { Add to client Width left and right windows frame widths }
  NewWidth := NewClWidth + (GetSystemMetrics(SM_CXFRAME) * 2 );
  { Add to client Height top and bottom windows frame widths plus the height
    of the window caption.  Adjust for border height }
  NewHeight := NewClHeight + (GetSystemMetrics(SM_CYFRAME) * 2 ) +
                             (GetSystemMetrics(SM_CYCAPTION) -
                              GetSystemMetrics(SM_CYBORDER));
  SavActive := FActive;               { Save Current Active State }
  FActive := False;                   { Ensure Messages NOT intercepted }
  { Ensure Minimum bounds reduce as well if necessary }
  if (NewWidth < MinWidth) then
    FMinWidth := NewWidth;            { Avoid Checking }

  if (NewHeight < MinHeight) then
    FMinHeight := NewHeight;          { Avoid Checking }

  { Set New Bounds for Object Inspector }
  MaxWidth := NewWidth;
  MaxHeight := NewHeight;

  FActive := SavActive;                { Restore Active State }
end;

(*
**  The procedure that does all the real work.  This modifies the windows
**  WM_GETMINMAXINFO message to adjust the maximum allowable range of the
**  window border track frame.
*)
procedure TFormSizeLimit.AdjustMinMaxMessage(MMPointer: Pointer);
var
  MinMaxPtr: ^TMinMaxInfo;      { Declare a pointer to a TMINMXINFO structure }
begin
  MinMaxPtr := MMPointer;       { overlay structure on passed address }
  { Set the tracking bounds }
  MinMaxPtr^.ptMinTrackSize := FMinRange;
  MinMaxPtr^.ptMaxTrackSize := FMaxRange;
end;

(*
**  This procedure intercepts ALL messages from windows that are addressed to
**  the form we are dropped onto.  If it detects a WM_GETMINMAXINFO message
**  which is sent prior to a resize by dragging the border, or prior to a
**  maximise event, then it intercepts in and passes it to our procedure
**  AdjustMinMaxMessage to modify the bounds.
**  The message (whatever it is) is then passed on down the chain for windows/
**  Delphi and the Form to handle.
*)
procedure TFormSizeLimit.NWndProc(var Messg: TMessage);
  (*
  **  Sub function visible only to NWndProc.  This calls the original window
  **  message handler.
  *)
  procedure CallInherited;
  begin
    with Messg do
      Result := CallWindowProc(FOldWndHandler, FFormHandle, Msg, wParam, lParam);
  end;

begin
  with Messg do
  begin
    case Msg of
      WM_GETMINMAXINFO:
          { If component set to active, and we're NOT designing }
          if (FActive and not (csDesigning in ComponentState)) then
            AdjustMinMaxMessage(Pointer(lParam));
      WM_SIZE:
          { If we ARE designing, and we want callback of bounds from form.
            However, if the Size Message is to Minimize the window, WE DO NOT
            want to adjust the Max Height/Width settings. }
          if ((csDesigning in ComponentState) and (FAutoTrack) and
             (wParam = SIZE_RESTORED)) then
            SetNewSize(LoWord(lParam), HiWord(lParam));
    end;
  end;
  { Ensure we pass on ALL messages down the line to windows }
  CallInherited;
end;

(*
**  Procedure to hook our message handler infront of the window message handler
**  for the form we are dropped onto.
*)
procedure TFormSizeLimit.MakeNewWndProc(FormHandle: hWnd);
begin
  { Save old message handler pointer first }
  FOldWndHandler := Pointer(GetWindowLong(FormHandle, GWL_WNDPROC));
  { Setup Special wrapper for our message handler }
  FWndHandlerPtr:= MakeObjectInstance(NWndProc);
  if FWndHandlerPtr = nil then
    raise EOutOfResources.Create('Windows Resources exhausted');
  { Set our new message handler as THE message handler for the form }
  SetWindowLong(FormHandle, GWL_WNDPROC, Longint(FWndHandlerPtr));
end;

(*
**  Restores the original window message handler to before we hooked it
*)
procedure TFormSizeLimit.SetOldWndProc(FormHandle: hWnd);
begin
  SetWindowLong(FormHandle, gwl_WndProc, Longint(FOldWndHandler));
end;

(*
**  Sets up the Min TPoint structures for the message handling routines
*)
procedure TFormSizeLimit.SetMinPoints;
begin
  { Set windows Min message handler ranges }
  FMinRange.X := FMinWidth;
  FMinRange.Y := FMinHeight;
end;

(*
**  Sets up the Max TPoint structures for the message handling routines
*)
procedure TFormSizeLimit.SetMaxPoints;
begin
  { Set windows Max message handler ranges }
  FMaxRange.X := FMaxWidth;
  FMaxRange.Y := FMaxHeight;
end;

(*
**  Public Method to enable setting of Maximum bounds
*)
procedure TFormSizeLimit.SetMaxSize(MaxBounds: TPoint);
begin
  FMaxRange := MaxBounds;
end;

(*
**  Public Method to enable setting of Minimum bounds
*)
procedure TFormSizeLimit.SetMinSize(MinBounds: TPoint);
begin
  FMinRange := MinBounds;
end;

(*
**  Registration of us for the Object Inspector & Component Palette
*)
procedure Register;
begin
  RegisterComponents('Custom', [TFormSizeLimit]);
end;

end.
